# -*- coding: utf-8 -*-
from odoo import models, api

class PosOrder(models.Model):
    _inherit = 'pos.order'


    def action_reset_to_draft_custom(self):
        """
        Reset POS Order to draft state. 
        Optionally deletes invoices and pickings based on configuration.
        """
        for order in self:
            # Skip if already draft to avoid unnecessary operations
            if order.state == 'draft':
                continue

            # 1. Handle Invoice (Account Move)
            if order.account_move:
                # In Odoo 18, we must first set to draft before unlinking
                if order.account_move.state != 'draft':
                    order.account_move.button_draft()
                
                # Delete the invoice
                order.account_move.with_context(force_delete=True).unlink()

            # 3. Supprimer ou invalider les pickings
            for picking in order.picking_ids:
                # if picking.state == 'done':
                #     continue  # Skip ceux qui sont déjà terminés
                for move in picking.move_ids:
                    for move_line in move.move_line_ids:
                        move_line.write({'quantity': 0.0})
                if picking.state not in ['done', 'cancel']:
                    picking.action_done()

            # 3. Force State Change to Draft (Bypass ORM Constraints)
            # The standard write() method raises 'This order has already been paid' on paid orders.
            # We use SQL injection to force the state to 'draft', which bypasses the python constraint.
            self.env.cr.execute("UPDATE pos_order SET state='draft' WHERE id=%s", (order.id,))
            order.invalidate_recordset(['state'])

            # 4. Handle Payments
            # Now that the order is legally 'draft' in the DB, we can try to unlink payments.
            # If the session is closed, standard unlink might still block. 
            # Depending on strictness, we might need SQL here too, but let's try ORM first.
            try:
                order.payment_ids.unlink()
            except Exception:
                # Fallback: detach payments if they cannot be deleted (e.g. closed session)
                order.write({'payment_ids': [(5, 0, 0)]})

            # 5. Clean up references via ORM
            # Remove link to deleted invoice
            order.write({
                'account_move': False,
            })

        return True

    def action_reset_to_in_payment(self):
        for order in self:
            # 1. Supprimer les paiements POS
            order.write({'state': 'draft'})
            for payment in order.payment_ids:
                if payment.account_move_id:
                    move = payment.account_move_id
                    for line in move.line_ids:
                        (line.matched_debit_ids + line.matched_credit_ids).unlink()
                    if move.state == 'posted':
                        move.button_draft()
                    move.unlink()
                payment.unlink()

            # 2. Supprimer la facture liée
            invoice = order.account_move
            if invoice:
                if invoice.state == 'posted':
                    invoice.button_draft()
                invoice.unlink()

            # 3. Supprimer ou invalider les pickings
            for picking in order.picking_ids:
                # if picking.state == 'done':
                #     continue  # Skip ceux qui sont déjà terminés
                for move in picking.move_ids:
                    for move_line in move.move_line_ids:
                        move_line.write({'quantity': 0.0})
                if picking.state not in ['done', 'cancel']:
                    picking.action_done()

            # 4. Remettre la commande à l'état brouillon
            order.write({'state': 'draft'})
    def reset_specific_payment_line(self, payment_line_id):
        payment = self.env['pos.payment'].browse(payment_line_id)
        if payment and payment.pos_order_id == self:
            if payment.account_move_id:
                move = payment.account_move_id
                for line in move.line_ids:
                    (line.matched_debit_ids + line.matched_credit_ids).unlink()
                if move.state == 'posted':
                    move.button_draft()
                move.unlink()
            payment.unlink()
        return True
        
class PosPayment(models.Model):
    _inherit = 'pos.payment'

    def button_reset_payment_line(self):
        for payment in self:
            payment.pos_order_id.reset_specific_payment_line(payment.id)